<?php

namespace App\DataTables;

use App\Facades\UtilityFacades;
use App\Models\TravelJourney;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class ProductsDataTable extends DataTable
{
    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        return datatables()
            ->eloquent($query)


            ->addColumn('image', function (TravelJourney $user) {
                return '<img src="'.Storage::url('uploads/product/'.$user->image).'" class="w-50">';
            })
            ->addColumn('price', function (TravelJourney $user) {
                return round($user->price *  $user->qty, 2);
            })

            ->editColumn('created_at', function (TravelJourney $user) {
                return UtilityFacades::date_time_format($user->created_at);
            })
            ->addColumn('action', function (TravelJourney $product) {
                return view('products.action', compact('product'));
            })

            ->rawColumns(['image', 'action']);
    }

    /**
     * Get query source of dataTable.
     *
     * @param \App\User $model
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(TravelJourney $model, Request $request)
    {

        $users =  $model->newQuery();
        if ($request->username) {
            $users->where('name', 'like', '%' . $request->username . '%');
        }
        if ($request->mobile_number) {
            $users->where('price', 'like', '%' . $request->mobile_number . '%');
        }
        if ($request->r_date) {
            // dd($request->r_date);
            $users->whereDate('created_at', '=', $request->r_date);
        }
        return $users;
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {
        return $this->builder()
            ->setTableId('users-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(1)
            ->language([
                "paginate" => [
                    "next" => '<i class="fas fa-angle-right"></i>',
                    "previous" => '<i class="fas fa-angle-left"></i>'
                ]
            ])
            ->parameters([
                "scrollX" => true
            ]);
    }

    /**
     * Get columns.
     *
     * @return array
     */
    protected function getColumns()
    {
        return [

            Column::make('id')->title('ProductID'),
            Column::make('image'),
            Column::make('name'),
            Column::make('price'),
            Column::make('created_at')->title('Add Time'),
            

            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(200)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get filename for export.
     *
     * @return string
     */
    // protected function filename()
    // {
    //     return 'Users_' . date('YmdHis');
    // }
}
